"use client";
import React from "react";
import Image from "next/image";
import { CheckCircle } from "lucide-react";
import { motion } from "framer-motion";
import { useLocale } from "next-intl";

export type IraqItem = {
  title: string;
  description: string;
  icon?: React.ReactNode; // لو حابب تبدّل الأيقونة
};

export type IraqSectionProps = {
  title: string;
  items: IraqItem[];
  image: { src: string; alt?: string; width?: number; height?: number };
  flag: { src: string; alt?: string; width?: number; height?: number };
  imageLeft?: boolean;
  className?: string;
};

export default function IraqSection({
  title,
  items,
  image,
  flag,
  imageLeft = true,
  className = "",
}: IraqSectionProps) {
  const locale = useLocale();
  const isArabic = locale === "ar";

  const orderClasses = imageLeft
    ? "lg:grid-cols-[1.05fr_1fr]"
    : "lg:grid-cols-[1fr_1.05fr] lg:[&>div:first-child]:order-2";

  return (
    <section className={`my-[7rem] ${className}`}>
      <motion.h2
        initial={{ opacity: 0, y: 10 }}
        whileInView={{ opacity: 1, y: 0 }}
        viewport={{ once: true }}
        transition={{ duration: 0.35 }}
        className="text-3xl sm:text-4xl font-extrabold tracking-tight text-secondary mb-4"
      >
        {title}
      </motion.h2>
      <hr className="border-slate-200 mb-2 md:w-[50%]" />

      <div className={`grid grid-cols-1 ${orderClasses} gap-10 items-center`}>
        <div className="flex flex-col text-start">
          <div className="mt-2 space-y-8">
            {items.map((item, idx) => (
              <motion.div
                key={idx}
                initial={{ opacity: 0, y: 8 }}
                whileInView={{ opacity: 1, y: 0 }}
                viewport={{ once: true }}
                transition={{ duration: 0.25, delay: idx * 0.05 }}
              >
                <div className="flex items-center gap-3">
                  <span className="text-teal-700">
                    {item.icon ?? (
                      <CheckCircle className="h-6 w-6" aria-hidden />
                    )}
                  </span>
                  <h3 className="text-lg md:text-xl font-bold tracking-wide text-primary">
                    {item.title}
                  </h3>
                </div>
                <p className="mt-2 text-[15px] sm:text-base leading-relaxed text-slate-600">
                  {item.description}
                </p>
              </motion.div>
            ))}
          </div>
        </div>

        <div className="relative">
          <motion.div
            initial={{ opacity: 0, y: 12 }}
            whileInView={{ opacity: 1, y: 0 }}
            viewport={{ once: true }}
            transition={{ duration: 0.35 }}
            className="overflow-hidden rounded-lg"
          >
            <Image
              src={image.src}
              alt={image.alt || "Iraq Shipping"}
              width={image.width || 980}
              height={image.height || 640}
              className="w-full h-auto object-cover rounded-lg"
              priority
            />
          </motion.div>

          <div
            className={`absolute -bottom-8 ${
              isArabic ? "left-8" : "right-8"
            } rounded-md overflow-hidden shadow-lg`}
          >
            <Image
              src={flag.src}
              alt={flag.alt || (isArabic ? "علم العراق" : "Iraq Flag")}
              width={flag.width || 180}
              height={flag.height || 112}
              className="w-[160px] sm:w-[180px] h-auto"
            />
          </div>
        </div>
      </div>
    </section>
  );
}
