// src/lib/meta.ts
import { Metadata } from "next";
import { OpenGraph } from "next/dist/lib/metadata/types/opengraph-types";
import { getLocale, getTranslations } from "next-intl/server";
import { Locale } from "next-intl";
import { locales } from "./config";

export const siteConfig = {
  title: "Fast Express",
  description: {
    en: "Fast Express is a leading delivery and logistics company providing fast and secure shipping services across road, air, and sea. We ensure reliable solutions for individuals and businesses with competitive prices and real-time tracking.",
    ar: "فاست إكسبريس هي شركة رائدة في خدمات التوصيل والشحن السريع عبر البر والجو والبحر. نضمن حلولاً موثوقة للأفراد والشركات مع أسعار تنافسية وخدمة تتبع فورية.",
  },
  keywords: {
    en: "Fast Express, delivery, shipping, logistics, courier, road shipping, air shipping, sea shipping, express delivery, tracking",
    ar: "فاست إكسبريس, توصيل, شحن, لوجستيات, بريد سريع, شحن بري, شحن جوي, شحن بحري, توصيل سريع, تتبع",
  },
  siteName: "Fast Express",
  image: "/images/logo.png",
  url: "https://fast-express.com",
};

export const metaObject = async (
  titleKey?: string,
  descriptionKey?: string,
  openGraph?: OpenGraph,
  overrideImage?: string,
  overrideUrl?: string
): Promise<Metadata> => {
  const [t, locale] = await Promise.all([
    getTranslations("METADATA"),
    getLocale(),
  ]);

  const currentLocale = locale as Locale;

  // Dynamic title
  const fullTitle = titleKey
    ? `${siteConfig.title} - ${titleKey}`
    : siteConfig.title;

  // Dynamic description
  const translatedDescription =
    descriptionKey ||
    siteConfig.description[
      currentLocale as keyof typeof siteConfig.description
    ];

  // Dynamic keywords
  const translatedKeywords =
    siteConfig.keywords[currentLocale as keyof typeof siteConfig.keywords];

  // Canonical URL
  const canonicalUrl = overrideUrl
    ? new URL(`/${locale}${overrideUrl}`, siteConfig.url).toString()
    : new URL(`/${locale}`, siteConfig.url).toString();

  // hreflang links
  const hreflangLinks = locales.map((loc) => ({
    href: `${siteConfig.url}/${loc}${overrideUrl || ""}`,
    hrefLang: loc,
  }));

  // Manifest
  const manifestPath = `/manifest/${currentLocale}.json`;

  return {
    metadataBase: new URL(siteConfig.url),
    alternates: {
      canonical: canonicalUrl,
      languages: {
        ...hreflangLinks.reduce<Record<string, string>>(
          (acc, { href, hrefLang }) => {
            acc[hrefLang] = href;
            return acc;
          },
          {}
        ),
      },
    },
    title: fullTitle,
    description: translatedDescription,
    keywords: translatedKeywords,
    robots: {
      index: true,
      follow: true,
      googleBot: {
        index: true,
        follow: true,
      },
    },
    icons: {
      icon: "/logo.png",
      shortcut: "/logo.png",
      apple: "/logo.png",
      other: {
        rel: "logo",
        url: "/logo.png",
      },
    },
    openGraph: {
      ...openGraph,
      title: fullTitle,
      description: translatedDescription,
      url: canonicalUrl,
      siteName: siteConfig.siteName,
      images: [
        {
          url: overrideImage || siteConfig.image,
          alt: "Fast Express Logo",
        },
      ],
      locale,
      type: "website",
    },
    twitter: {
      card: "summary_large_image",
      title: fullTitle,
      description: translatedDescription,
      images: [
        {
          url: overrideImage || siteConfig.image,
          alt: "Fast Express Logo",
        },
      ],
    },
    applicationName: siteConfig.siteName,
    other: {
      "mobile-web-app-capable": "yes",
      manifest: manifestPath,
    },
    formatDetection: {
      telephone: true,
    },
    appleWebApp: {
      title: fullTitle,
      capable: true,
      statusBarStyle: "default",
      startupImage: "/images/logo.png",
    },
  };
};
